//
// Radium Exporter.js
//
// v.20070130
//  required version : Cheetah3D v3.5
//
// 20061229 first.
// 20070117 fixed bugs. added procedural.
// 20070117 fixed small bug.
// 20070130 fixed normal culculation bug.
//
// @author Hiroto Tsubaki (tg@tres-graficos.jp)
// special thanks to Martin, Peter, and Yuichi Masuda. thanks to Ian.
//
// Description: Export Scene for Radium (.scn file).
// Usage: Place this into ~/Library/Application Support/Cheetah3D/scripts/Tool folder. restart Cheetah3D, then select from Tools -> Script -> Tool Script
//
// this from Todd's script
// add function to Array class.
if( !Array.indexOf) {
    Array.prototype.indexOf = function(str) {
        for(var i = 0; i < this.length; i++) {
            if(this[i] == str) {
                return i;
            }
        }
        return -1;
    }
}
//
var SCN_FILE; // File object.
var SCN_PATH = '';

var radium_ver = '0.00.09';
var radium_cmd = "java -server -Xmx800m -jar /Applications/Radium/bin/Radium.jar -threads 16"; // radium.jar path

var SHAPE_COUNT = 0;
var MAT_COUNT = 0;
var LIGHT_COUNT = 0;

var DEF_MAT_NAME = 'CM_dif_def';

var PRIMITIVE_LIST = new Array; // store primitive (Ball, Cube .etc.)
var MESH_LIST = new Array; // store polygon object
var BOOLEAN_LIST = new Array; // for CSG
var LIGHT_LIST = new Array; // store lights
var MAT_LIST = new Array; // store materials

var IMG_PATHS = new Array;
var IMG_NAMES = new Array;

var SUN_LIGHT = undefined;

var Normal_List = new Array(new Vec3D(1, 0, 0), new Vec3D(0, 1, 0), new Vec3D(0, 0, 1));
var Light_Normals = new Array(new Vec3D(-1, 0, 0), new Vec3D(0, -1, 0), new Vec3D(0, 0, -1));
var Sky_Types = new Array('constant', 'emit');
var Light_Types = new Array('constant', 'constant', 'emit');
var CSG_Operations = new Array(' + ', ' - ', ' & ', ' / ', ' | ');

var Procedurals = new Array;
Procedurals['noise'] = 0;
Procedurals['checker'] = 0;
Procedurals['bump'] = 1;
Procedurals['dent'] = 1;
Procedurals['pyramid'] = 1;
Procedurals['quilt'] = 0;
Procedurals['gradient'] = 0;
Procedurals['stripe'] = 0;

function buildUI(tool) {
    tool.addParameterSeparator("Radium Exporter");
    
    tool.addParameterBool("separate mesh file",0,0,1,true,true);
    tool.addParameterFloat("point light radius",0.05,0,100,true,true);
    tool.addParameterBool("use primitive",1,0,1,true,true);
    tool.addParameterBool("primitives with uv",1,0,1,true,true);
    tool.addParameterBool("use CSG",1,0,1,true,true);
    
    tool.addParameterSeparator("Settings");
    tool.addParameterBool("filter_samples",1,0,1,true,true);
    
    tool.addParameterSeparator("Background");
    
    tool.addParameterSelector("background type",Sky_Types,true,true);
    
    tool.addParameterSeparator("Physical Sky - [ distant light as sun ]");
    tool.addParameterFloat("sky_gain",0.002,0,1,true,true);
    tool.addParameterFloat("sun_gain",1.0,0,100,true,true);
    tool.addParameterFloat("turbidity",2.0,0,15,true,true);
    
    tool.addParameterSeparator("Export");
    tool.addParameterButton("Export","export","exportSCN");
    //tool.addParameterSeparator("Export & Run");
    //tool.addParameterButton("Export & Run","export & run","exportSCNAndRun");
}

function shapeName(name) {
    var names = name.split(" ").join("");
    SHAPE_COUNT += 1;
    return "CH_"+names+"_"+SHAPE_COUNT;
}

function matName(type, name) {
    var names = name.split(" ").join("");
    MAT_COUNT += 1;
    return "CM_"+type+"_"+names+"_"+MAT_COUNT;
}

function lightName(name) {
    var names = name.split(" ").join("");
    LIGHT_COUNT += 1;
    return "CL_"+names+"_"+LIGHT_COUNT;
}

function imgName(type, picname) {
    var names = picname.split(" ").join("");
    names = names.split(".");
    return "CT_"+type+"_"+names[0];
}

function filenameForTexture(filepath) {
    if (filepath == "none") return false;
    var file = new File(filepath);
    return file.lastPathComponent();
}

function tagOfObject(obj, tagType) {
    var i;
    var tagCount = obj.tagCount();
    var hasTag = false;
    for (i = 0;i < tagCount;i++) {
        var tag = obj.tagAtIndex(i);
        if (tag.type() == tagType) {
            hasTag = true;
            returnTag = tag;
        }
    }
    return (hasTag)? returnTag : false;
}

function vector3Line(vec) {
    return "<"+vec.x.toPrecision(6)+", "+vec.y.toPrecision(6)+", "+vec.z.toPrecision(6)+">";
}

function vector4Line(vec) {
    return "<"+vec.r.toPrecision(6)+", "+vec.g.toPrecision(6)+", "+vec.b.toPrecision(6)+", "+vec.a.toPrecision(6)+">";
}

function storeObjects(obj, rot, scale, tool) {
    var obj_rot = obj.getParameter("rotation");
    var obj_scale = obj.getParameter("scale");
    var use_prim = tool.getParameter("use primitive");
    var use_csg = tool.getParameter("use CSG");
    //
    rot = rot.add(obj_rot);
    scale = new Vec3D(obj_scale.x*scale.x, obj_scale.y*scale.y, obj_scale.z*scale.z);
    //
    if (obj.family() == NGONFAMILY) {
        if (obj.type() == BALL && use_prim) {
            PRIMITIVE_LIST.push(new Array(obj, rot, scale));
        } else if (obj.type() == 4 && use_prim) { //CYLINDER
            PRIMITIVE_LIST.push(new Array(obj, rot, scale));
        } else if (obj.type() == CONE && use_prim) {
            PRIMITIVE_LIST.push(new Array(obj, rot, scale));
        } else if (obj.type() == BOX&& use_prim) {
            PRIMITIVE_LIST.push(new Array(obj, rot, scale));
        } else if (obj.type() == PLANE && use_prim) {
            PRIMITIVE_LIST.push(new Array(obj, rot, scale));
        } else if (obj.type() == DISC && use_prim) {
            PRIMITIVE_LIST.push(new Array(obj, rot, scale));
        } else if (obj.type() == TORUS && use_prim) {
            PRIMITIVE_LIST.push(new Array(obj, rot, scale));
        } else if (obj.type() == BOOLEAN && use_csg) {
            BOOLEAN_LIST.push(new Array(obj, rot, scale));
        } else { // Polygn object or other Prameterics
            MESH_LIST.push(new Array(obj, rot, scale));
        }
    } else if (obj.family() == LIGHTFAMILY) {
        if (obj.getParameter("lightType") == 2) SUN_LIGHT = obj; // treating distant light as sun. 
        else LIGHT_LIST.push(new Array(obj, rot, scale));
    }
    if (obj.isCeatorObj() == false) {
        var i = 0;
        var childCount = obj.childCount();
        for (i = 0;i < childCount;i++) {
            storeObjects(obj.childAtIndex(i), rot, scale, tool);
        }
    }
}

function writeCaption(str) {
    SCN_FILE.writeln("/* -------------------------------------------------");
    SCN_FILE.writeln("   -----  "+str);
    SCN_FILE.writeln("   ------------------------------------------------- */\n");
}

function exportSCNAndRun(tool) { //
    exportSCN(tool);
    path = radium_cmd+" "+SCN_PATH;
    OS.system(radium_cmd+" "+path);
}

function exportSCN(tool) {
    var doc = tool.document();
    // get path for SCN file
    var path = OS.runSavePanel("scn");
    if (path == null) {
        return;
    }
    SCN_PATH = path; // store path
    
    // initialize
    PRIMITIVE_LIST = new Array();
    MESH_LIST = new Array();
    BOOLEAN_LIST = new Array();
    //
    IMG_NAMES = new Array();
    IMG_PATHS = new Array();
    //
    LIGHT_LIST = new Array();
    MAT_LIST = new Array();
    SUN_LIGHT = undefined;
    
    SHAPE_COUNT = 0;
    MAT_COUNT = 0;
    LIGHT_COUNT = 0;
    //
    // print("---- SCN Export.js ----");
    SCN_FILE = new File(path);
    SCN_FILE.open(WRITE_MODE);
    //
    writeCaption("radium scene description language file\n"+"   -----  for radium v."+radium_ver+"\n   -----  created by Cheetah3D");
    // storing objects, meshs and lights
    storeObjects(doc.root(), new Vec3D(0,0,0), new Vec3D(1,1,1), tool);
    // settings
    writeCaption("settings");
    writeSetting(tool);
    // material
    writeCaption("materials");
    SCN_FILE.writeln("material "+DEF_MAT_NAME+" = {\n\tdiffuse 0.8\n}\n");
    var matCount = doc.materialCount();
    for (i = 0;i < matCount;i++) {
        var material = doc.materialAtIndex(i);
        var mat_name = writeMaterial(material, [""], "block");
        MAT_LIST.push(mat_name);
    }
    // camera and sky
    writeCaption("camera and sky");
    writeCamera(tool);
    // lights
    var lightCount = LIGHT_LIST.length;
    if (lightCount) writeCaption("lights");
    for (i = 0;i < lightCount;i++) {
        writeLight(LIGHT_LIST[i], tool);
    }
    // primitives
    var primCount = PRIMITIVE_LIST.length;
    if (primCount) writeCaption("primitives");
    for (i = 0;i < primCount;i++) {
        writePrimitive(PRIMITIVE_LIST[i], tool, true);
    }
    // booleans
    var booleanCount = BOOLEAN_LIST.length;
    if (booleanCount) writeCaption("csg");
    for (i = 0;i < booleanCount;i++) {
        writeBoolean(BOOLEAN_LIST[i], tool, true);
    }
    // meshs (polygon object)
    var meshCount = MESH_LIST.length;
    if (meshCount) writeCaption("mesh");
    for (i = 0;i < meshCount;i++) {
        writeMesh(MESH_LIST[i], tool, true);
    }
    //
    SCN_FILE.close();
}

function writeCamera(tool) {
    var doc = tool.document();
    var cam = doc.activeCamera();
    var fov = cam.getParameter("fieldOfView") * 0.018;
    var rot = cam.getParameter("rotation");
    var mat = new Mat4D(ROTATE_HPB, rot.x, rot.y, rot.z);

    var pos = cam.getParameter("position");
    var xaxis = new Vec3D(1,0,0);
    var yaxis = new Vec3D(0,1,0);
    var zaxis = new Vec3D(0,0,-1);

    xaxis = mat.multiply(xaxis);
    yaxis = mat.multiply(yaxis);
    zaxis = mat.multiply(zaxis);
    
    SCN_FILE.write("\ndefine_camera {\n");
    SCN_FILE.write("\tfov "+fov.toPrecision(3)+"\n");
    SCN_FILE.write("\tlocation "+vector3Line(pos)+"\n");
    SCN_FILE.write("\txaxis "+vector3Line(xaxis)+"\n");
    SCN_FILE.write("\tyaxis "+vector3Line(yaxis)+"\n");
    SCN_FILE.write("\tzaxis "+vector3Line(zaxis)+"\n");
    SCN_FILE.write("}\n");
    
    var hasSun = false;
    var sky_type = tool.getParameter("background type");
    // sky setting
    if (SUN_LIGHT) {
        hasSun = true;
        var sun_mat = SUN_LIGHT.obj2WorldMatrix();
        var pos = new Vec3D(sun_mat.m03, sun_mat.m13, sun_mat.m23);
        var sky_gain = tool.getParameter("sky_gain");
        var sun_gain = tool.getParameter("sun_gain");
        var turbidity = tool.getParameter("turbidity");
        if (pos.y > 0) {
            SCN_FILE.writeln("\nsun {");
            SCN_FILE.writeln("\tsun_position "+vector3Line(pos));
            if (sky_gain > 0) SCN_FILE.writeln("\tsky_gain "+sky_gain.toPrecision(6));
            if (sun_gain > 0) SCN_FILE.writeln("\tsun_gain "+sun_gain.toPrecision(6));
            if (turbidity > 0) SCN_FILE.writeln("\tturbidity "+turbidity.toPrecision(6));
            SCN_FILE.writeln("}\n");
        } else {
            hasSun = false;
        }
    }
    if (! hasSun) {
        var cam_hdri = tagOfObject(cam,HDRITAG);
        var cam_bgImage = '';
        if (cam_hdri != false && cam_hdri.getParameter("hdriBackgroundImage") != "none") {
            var hdri = cam_hdri.getParameter("hdriBackgroundImage");
            cam_bgImage = filenameForTexture(hdri);
            cam_bgPower = cam_hdri.getParameter("hdriPower");
        }
        if (cam_bgImage) {
            SCN_FILE.write("\nimage chs_sky = {file(\""+cam_bgImage+"\")}\n");
            SCN_FILE.write("set_sky {\n");
            SCN_FILE.write("\t"+Sky_Types[sky_type]+" "+cam_bgPower.toPrecision(6)+"\n");
            SCN_FILE.write("\tmap image {chs_sky}\n");
            SCN_FILE.write("}\n\n");
        } else {
            var sky_color = cam.getParameter("backgroundColor");
            SCN_FILE.write("\nset_sky {\n");
            SCN_FILE.write("\t"+Sky_Types[sky_type]+" "+vector3Line(sky_color)+"\n");
            SCN_FILE.write("}\n\n");
        }
    }
}

function writeSetting(tool) {
    var doc = tool.document();
    var cam = doc.activeCamera();
    
    var img_width = cam.getParameter("resolutionX");
    var img_height = cam.getParameter("resolutionY");
    
    var filter_samples = tool.getParameter("filter_samples");
    // you can custmize radium settings here.
    SCN_FILE.writeln("settings {");
    SCN_FILE.writeln("\timage_width "+img_width);
    SCN_FILE.writeln("\timage_height "+img_height);
    SCN_FILE.writeln("\tmax_ray_depth 5");
    SCN_FILE.writeln("\tgi_max_depth 3");
    
    if (filter_samples) SCN_FILE.writeln("\n\tfilter_samples true");
    
    SCN_FILE.writeln("}\n");
}

function squareScale(scale, type, v1, v2) {
    if (type == 0) { // X
        scale.z = scale.z * v1;
        scale.x = scale.x * v2;
    } else if (type == 1) { // Y
        scale.x = scale.x * v1;
        scale.z = scale.z * v2;
    } else if (type == 2) { // Z
        scale.z = scale.z * v1;
        scale.x = scale.x * v2;
    }
    return scale;
}

function lightScale(scale, type, v1, v2) {
    if (type == 0) { // X
        scale.z = scale.z * v1;
        scale.x = scale.x * v2;
    } else if (type == 1) { // Y
        scale.x = scale.x * v2;
        scale.z = scale.z * v1;
    } else if (type == 2) { // Z
        scale.z = scale.z * v1;
        scale.x = scale.x * v2;
    }
    return scale;
}

function writeLight(light_data, tool) {
    var light = light_data[0];
    if (light.family() == LIGHTFAMILY) {
        var i,j;
        var light_name = lightName(light.getParameter("name"));
        var light_mat = light.obj2WorldMatrix();
        var pos = new Vec3D(light_mat.m03, light_mat.m13, light_mat.m23); // from the matrix of object.
        var rot = light_data[1];
        var scale = light_data[2];
        var light_type = light.getParameter("lightType");
        var light_intensity = light.getParameter("intensity");
        var light_color = light.getParameter("color");
        var light_atten = light.getParameter("attenuation");
        light_color = light_color.multiply(light_intensity);
        switch(parseInt(light_type)) {
            case 3: // Point
                var radius = tool.getParameter("point light radius");
                SCN_FILE.writeln("shape "+light_name+" = sphere {");
                SCN_FILE.writeln("\t<0,0,0>, "+radius.toPrecision(5));
                SCN_FILE.writeln("\tmaterial {"+Light_Types[light_atten]+" "+vector3Line(light_color)+"}");
                writeMatrix(SCN_FILE, pos, scale, rot);
                SCN_FILE.writeln("\tno_shadow");
                SCN_FILE.writeln("}\nadd_shape "+light_name+"\n");
                break;
            case 1: // Area
                var axis = light.getParameter("axis");
                var light_width = light.getParameter("width");
                var light_height = light.getParameter("height");
                SCN_FILE.writeln("shape "+light_name+" = square {");
                SCN_FILE.writeln("\t<0,0,0>, "+vector3Line(Light_Normals[axis])+", 0.5, 0");
                SCN_FILE.writeln("\tmaterial {"+Light_Types[light_atten]+" "+vector3Line(light_color)+"}");
                scale = lightScale(scale, axis, light_width, light_height);
                writeMatrix(SCN_FILE, pos, scale, rot);
                SCN_FILE.writeln("\tno_shadow");
                SCN_FILE.writeln("}\nadd_shape "+light_name+"\n");
                break;
        }
    }
}

function writePrimitive(obj_data, tool, add) {
    var uved = tool.getParameter("primitives with uv");
    var obj = obj_data[0];
    if (obj.family() == NGONFAMILY) {
        var shape = obj.type();
        var i,j;
        var obj_name = shapeName(obj.getParameter("name"));
        var obj_mat = obj.obj2WorldMatrix(); // this affects only position parameter. 
        var pos = new Vec3D(obj_mat.m03, obj_mat.m13, obj_mat.m23); //from the matrix of object.
        var rot = obj_data[1]; //
        var scale = obj_data[2]; //
        var obj_matlist = obj.materialTags();
        rot = rot.multiply(-1);
        switch(shape) {
            case BALL:
                var radius = obj.getParameter("radius");
                SCN_FILE.writeln("shape "+obj_name+" = sphere {");
                SCN_FILE.writeln("\t<0,0,0>, "+radius.toPrecision(4));
                writeMatrix(SCN_FILE, pos, scale, rot);
                break;
            case 4: //CYLINDER
                var radius = obj.getParameter("radius");
                var height = obj.getParameter("height");
                scale.y = scale.y * height;
                SCN_FILE.writeln("shape "+obj_name+" = cylinder {");
                SCN_FILE.writeln("\t<0,-0.5,0>, <0,0.5,0>, "+radius.toPrecision(4));
                writeMatrix(SCN_FILE, pos, scale, rot);
                break;
            case CONE:
                var radius = obj.getParameter("radius");
                var height = obj.getParameter("height");
                scale.y = scale.y * height;
                var move = new Vec3D(0, height / 2, 0);
                var moveR = new Mat4D(ROTATE, -rot.x, -rot.y, -rot.z);
                var moveS = new Mat4D(SCALE, scale.x, scale.y, scale.z);
                move = moveR.multiply(moveS.multiply(move));
                SCN_FILE.writeln("shape "+obj_name+" = cone {");
                SCN_FILE.writeln("\t<0,-0.5,0>,<0,0.5,0>, "+radius.toPrecision(4));
                writeMatrix(SCN_FILE, pos, scale, rot);
                SCN_FILE.writeln("\ttranslate "+vector3Line(move));
                break;
            case BOX:
                var width = obj.getParameter("sizeX");
                var height = obj.getParameter("sizeY");
                var length = obj.getParameter("sizeZ");
                scale.x = scale.x * width;
                scale.y = scale.y * height;
                scale.z = scale.z * length;
                SCN_FILE.writeln("shape "+obj_name+" = box {");
                SCN_FILE.writeln("\t<0.5,0.5,0.5>, <-0.5,-0.5,-0.5>");
                writeMatrix(SCN_FILE, pos, scale, rot);
                break;
            case PLANE:
                var ori = obj.getParameter("orientation");
                var width = obj.getParameter("sizeX");
                var height = obj.getParameter("sizeY");
                SCN_FILE.writeln("shape "+obj_name+" = square {");
                SCN_FILE.writeln("\t<0,0,0>, "+vector3Line(Normal_List[ori])
                    +", 0.5, 0");
                scale = squareScale(scale, ori, width, height);
                writeMatrix(SCN_FILE, pos, scale, rot);
                break;
            case DISC:
                var radius = obj.getParameter("radius");
                var normal = Normal_List[1];
                SCN_FILE.writeln("shape "+obj_name+" = disc {");
                SCN_FILE.writeln("\t<0,0,0>, "+vector3Line(normal)
                    +", "+radius.toPrecision(4)+", 0");
                writeMatrix(SCN_FILE, pos, scale, rot);
                break;
            case TORUS:
                var radius_ring = obj.getParameter("radiusRing");
                var radius_tube = obj.getParameter("radiusTube");
                SCN_FILE.writeln("shape "+obj_name+" = torus {");
                SCN_FILE.writeln("\t<0,0,0>, "
                    +radius_ring.toPrecision(4)+", "
                    +radius_tube.toPrecision(4));
                writeMatrix(SCN_FILE, pos, scale, rot);
                break;
        }
        writeMaterialBlock(obj_matlist, SCN_FILE, "");
        if (uved) SCN_FILE.writeln("\tdo_uv");
        SCN_FILE.writeln("}");
        if (add) SCN_FILE.writeln("add_shape "+obj_name+"\n");
        
        return obj_name;
    }
}

function writeMatrix(file, pos, scale, rot) {
        if (scale.x != 1 || scale.y != 1 || scale.z != 1) file.writeln("\tscale "+vector3Line(scale));
        if (pos.norm() != 0) file.writeln("\ttranslate "+vector3Line(pos));
        if (rot.x != 0) file.writeln("\troty "+rot.x.toPrecision(6));
        if (rot.y != 0) file.writeln("\trotx "+rot.y.toPrecision(6));
        if (rot.z != 0) file.writeln("\trotz "+rot.z.toPrecision(6));
}

function writeImageBlock(file, path) {
    var index = IMG_PATHS.indexOf(path);
    if (index < 0) {
        IMG_PATHS.push(path);
    } else {
        return IMG_NAMES[index];
    }
    var image_fname = filenameForTexture(path);
    if (image_fname) {
        var image_name = imgName('img',image_fname);
        file.writeln("image "+image_name+" = {file(\""+image_fname+"\")}\n");
        IMG_NAMES.push(image_name);
        return image_name;
    }
}

function writeMaterialBlock(matlist, file, mode) {
    var sels = new Array; // store shading selection.
    if (matlist.length >= 1) {
        if (mode == 'list') {
            var i;
            file.writeln("\tmaterial_list {\n\t\t"+matlist.length+",");
            for (i = 0;i < matlist.length;i++) {
                var mat_tag = matlist[i];
                var mat_index = mat_tag.linkedToMaterial();
                if (i < matlist.length - 1) file.writeln("\t\t"+MAT_LIST[mat_index]+",");
                else file.writeln("\t\t"+MAT_LIST[mat_index]);
                sels.push(mat_tag.getParameter("shadeSelection"));
            }
            file.write("\t}\n");
        } else {
            var mat_tag = matlist[0];
            var uv_scale = mat_tag.getParameter("UVScale");
            var uv_offset = mat_tag.getParameter("UVOffset");
            var uv_rotate = mat_tag.getParameter("UVRotation");
            var sOffset = mat_tag.getParameter("shadingOffset");
            var sRotation = mat_tag.getParameter("shadingRotation");
            var sScale = mat_tag.getParameter("shadingScale");
            if (uv_scale.u != 1 || uv_scale.v != 1 || sScale.x + sScale.y + sScale.z != 3 || sOffset.norm() != 0 || sRotation.norm() != 0) { // || uv_offset.norm() != 0 || uv_rotate != 0) {
                var doc = mat_tag.document();
                var material = doc.materialAtIndex(mat_tag.linkedToMaterial());
                writeMaterial(material, new Array(uv_offset,uv_scale,uv_rotate,sOffset,sScale,sRotation), "inline");
            } else {
                var mat_index = mat_tag.linkedToMaterial();
                file.writeln("\tmaterial "+MAT_LIST[mat_index]);
            }
        }
    } else {
        file.writeln("\tmaterial "+DEF_MAT_NAME);
    }
    return sels;
}

function writeMesh(obj_data, tool, add) {
    var obj = obj_data[0];
    if (obj.family() == NGONFAMILY) {
        var i,j;
        var obj_name = shapeName(obj.getParameter("name"));
        var obj_mat = obj.obj2WorldMatrix();
        var obj_matlist = obj.materialTags();
        var obj_matsels = new Array;
        var core = obj.modCore();
        var vertCount = core.vertexCount();
        var polyCount = core.polygonCount();
        var triCount = core.triangleCount();
        var obj_pos = new Vec3D(obj_mat.m03, obj_mat.m13, obj_mat.m23);
        var obj_rot = obj_data[1];
        var obj_scale = obj_data[2];
        var obj_info = new Array;
        obj_rot = obj_rot.multiply(-1);
        
        // polygon check
        if (triCount < 1) {
            var res = OS.messageBox("Polygon Alert - "+obj.getParameter("name"),
                "this object has no polygon. it may cause some error for radium.\nplease check a object.");
        }
        if (tool.getParameter("separate mesh file")) {
            var dir = SCN_FILE.directory();
            var file_name = SCN_FILE.lastPathComponent();
            var file_name_list = file_name.split(".");
            var mesh_filename = file_name_list[0] + '_' + obj_name + '.scn';
            var path = dir + '/' + mesh_filename;
            var file = new File(path);
            file.open(WRITE_MODE);
        } else {
            file = SCN_FILE;
        }
        
        if (obj_matlist.length < 1) {
            obj_matsels.push(new Array(0, DEF_MAT_NAME));
        } else {
            for (i = 0;i < obj_matlist.length; i++) {
                var mat_tag = obj_matlist[i];
                var sel = mat_tag.getParameter("shadeSelection");
                var uv_scale = mat_tag.getParameter("UVScale");
                var uv_offset = mat_tag.getParameter("UVOffset");
                var uv_rotate = mat_tag.getParameter("UVRotation");
                var sOffset = mat_tag.getParameter("shadingOffset");
                var sRotation = mat_tag.getParameter("shadingRotation");
                var sScale = mat_tag.getParameter("shadingScale");
                if (uv_scale.u != 1 || uv_scale.v != 1 || sScale.x + sScale.y + sScale.z != 3 || sOffset.norm() != 0 || sRotation.norm() != 0) { // || uv_offset.norm() != 0 || uv_rotate != 0) {
                    var doc = mat_tag.document();
                    var material = doc.materialAtIndex(mat_tag.linkedToMaterial());
                    var mat_iname = writeMaterial(material, new Array(uv_offset,uv_scale,uv_rotate,sOffset,sScale,sRotation), "block_extra");
                    obj_matsels.push(new Array(sel, mat_iname));
                } else {
                    obj_matsels.push(new Array(sel, MAT_LIST[mat_tag.linkedToMaterial()]));
                }
            }
        }
        // shape header.
        file.writeln("shape "+obj_name+" = mesh {\n");
        file.writeln("\tmaterial_list {\n\t\t"+obj_matsels.length+",");
        for (i = 0;i < obj_matsels.length;i++) {
            if (i < obj_matsels.length - 1) file.writeln("\t\t"+obj_matsels[i][1]+",");
            else file.writeln("\t\t"+obj_matsels[i][1]);
        }
        file.writeln("\t}");
        // vertex list
        file.write("\tvertex_list {\n\t\t"+vertCount);
        for (i = 0;i < vertCount;i++) {
            var vert = core.vertex(i);
            file.write(",\n");
            file.write("\t\t"+vector3Line(vert));
        }
        file.write("\n\t}\n");
        // face list
        file.write("\tface_indices {\n\t\t"+triCount);
        for (i = 0;i < polyCount;i++) {
            var polySize = core.polygonSize(i);
            var mat_num = 0;
            for (j = 0;j < obj_matsels.length;j++) {
                var matsel = obj_matsels[j][0] - 1;
                if (matsel < 0) { // all
                    mat_num = 0;
                } else { // check selcted
                    core.setActivePolygonSelection(matsel);
                    if (core.polygonSelection(i)) mat_num = j;
                }
            }
            core.setActivePolygonSelection(0);
            if (polySize > 3) {
                for (j = 0;j < polySize-2;j++) {
                    var nums = core.triangle(i,j);
                    var n1 = core.vertexIndex(i,nums[0]);
                    var n2 = core.vertexIndex(i,nums[2]);
                    var n3 = core.vertexIndex(i,nums[1]);
                    file.write(",\n\t\t<"+n1+", "+n2+", "+n3+">, "+mat_num);
                }
            } else {
                var n1 = core.vertexIndex(i,0);
                var n2 = core.vertexIndex(i,2);
                var n3 = core.vertexIndex(i,1);
                file.write(",\n\t\t<"+n1+", "+n2+", "+n3+">, "+mat_num);
            }
        }
        file.write("\n\t}\n");
        // face normal
        file.write("\tface_normal {\n\t\t"+triCount);
        for (i = 0;i < polyCount;i++) {
            var polySize = core.polygonSize(i);
            if (polySize > 3) {
                for (j = 0;j < polySize-2;j++) {
                    var nums = core.triangle(i,j);
                    
                    var norm1 = core.normal(i,nums[0]);
                    var norm2 = core.normal(i,nums[2]);
                    var norm3 = core.normal(i,nums[1]);

                    file.write(",\n\t\t"+vector3Line(norm1)+", "
                                +vector3Line(norm2)+", "
                                +vector3Line(norm3));
                }
            } else {
                
                var norm1 = core.normal(i,0);
                var norm2 = core.normal(i,2);
                var norm3 = core.normal(i,1);

                file.write(",\n\t\t"+vector3Line(norm1)+", "
                                +vector3Line(norm2)+", "
                                +vector3Line(norm3));
            }
        }
        file.write("\n\t}\n");
        // face uv
        file.write("\tface_uv {\n\t\t"+triCount);
        for (i = 0;i < polyCount;i++) {
            var polySize = core.polygonSize(i);
            if (polySize > 3) {
                for (j = 0;j < polySize-2;j++) {
                    var nums = core.triangle(i,j);
                    var uv1 = core.uvCoord(i,nums[0]);
                    var uv2 = core.uvCoord(i,nums[2]);
                    var uv3 = core.uvCoord(i,nums[1]);
                    file.write(",\n\t\t<"+uv1.u.toPrecision(6)+", "+uv1.v.toPrecision(6)
                                    +">, <"+uv2.u.toPrecision(6)+", "+uv2.v.toPrecision(6)
                                    +">, <"+uv3.u.toPrecision(6)+", "+uv3.v.toPrecision(6)+">");
                }
            } else {
                var uv1 = core.uvCoord(i,0);
                var uv2 = core.uvCoord(i,2);
                var uv3 = core.uvCoord(i,1);
                file.write(",\n\t\t<"+uv1.u.toPrecision(6)+", "+uv1.v.toPrecision(6)
                                +">, <"+uv2.u.toPrecision(6)+", "+uv2.v.toPrecision(6)
                                +">, <"+uv3.u.toPrecision(6)+", "+uv3.v.toPrecision(6)+">");
            }
        }
        file.writeln("\t}");
        writeMatrix(file, obj_pos, obj_scale, obj_rot);
        file.writeln("}\n");
        // footer
        if (add) file.writeln("\nadd_shape "+obj_name+"\n");
        
        if (tool.getParameter("separate mesh file")) {
            file.close();
            SCN_FILE.writeln("// "+obj.getParameter("name")+" as "+obj_name);
            SCN_FILE.writeln("#include \""+mesh_filename+"\"");
        }
        
        return obj_name;
    }
}

function writeBoolean(obj_data, tool, add) {
    var obj = obj_data[0];
    var rot = obj_data[1];
    var scale = obj_data[2];
    if (obj.type() == BOOLEAN) {
        var child_a = obj.childAtIndex(0);
        var child_b = obj.childAtIndex(1);
        var operation = obj.getParameter("operation");

        var rot_a = rot.add(child_a.getParameter("rotation"));
        var scale_a = child_a.getParameter("scale");
        scale_a = new Vec3D(scale_a.x*scale.x, scale_a.y*scale.y, scale_a.z*scale.z);
        //
        var rot_b = rot.add(child_b.getParameter("rotation"));
        var scale_b = child_b.getParameter("scale");
        scale_b = new Vec3D(scale_b.x*scale.x, scale_b.y*scale.y, scale_b.z*scale.z);
        //
        var name_a;
        var name_b;
        //
        if (child_a.type() == BALL || child_a.type() == 4 || child_a.type() == CONE || child_a.type() == BOX || child_a.type() == PLANE || child_a.type() == DISC) {
            name_a = writePrimitive(new Array(child_a, rot_a, scale_a), tool, false);
        } else if (child_a.type() == BOOLEAN) {
            name_a = writeBoolean(new Array(child_a, rot_a, scale_a), tool, false);
        } else if (child_a.family() == NGONFAMILY) {
            name_a = writeMesh(child_a, tool, false);
        }
        if (child_b.type() == BALL || child_b.type() == 4 || child_b.type() == CONE || child_b.type() == BOX || child_b.type() == PLANE || child_b.type() == DISC) {
            name_b = writePrimitive(new Array(child_b, rot_b, scale_b), tool, false);
        } else if (child_b.type() == BOOLEAN) {
            name_b = writeBoolean(new Array(child_b, rot_b, scale_b), tool, false);
        } else if (child_b.family() == NGONFAMILY) {
            name_b = writeMesh(child_b, tool, false);
        }
        
        var obj_name = shapeName(obj.getParameter("name"));
        SCN_FILE.writeln("\nshape "+obj_name+" = "+name_a+CSG_Operations[operation]+name_b);
        if (add) SCN_FILE.writeln("add_shape "+obj_name);
        
        return obj_name;
    }
}

function writeMaterial(material, info, mode) {
    var mat_name = material.getParameter("name");
    var mat_type = material.getParameter("shaderType");
    var matrix_line = '';
    var indent = (mode == 'inline')? "\t" : "";
    var mat_infos = mat_name.split("_");
    var mat_ior_e = (mat_infos.length > 1)? parseFloat(mat_infos.pop()) : undefined;

    if (mode == 'inline' || mode == 'block_extra') {
        var uscale = 1 / info[1].u;
        var vscale = 1 / info[1].v;
        matrix_line = " uscale "+uscale.toPrecision(6)+" vscale "+vscale.toPrecision(6);
    }
    //
    switch (mat_type) {
        case "Material":
            var colorC = material.getParameter("colorColor");
            var colorT = material.getParameter("colorTex");
            var defIntensity = material.getParameter("diffIntensity");
            var colorMix = material.getParameter("colorMix");
            var specC = material.getParameter("specularColor");
            var specT = material.getParameter("specTex");
            var specMix = material.getParameter("specMix");
            var specSize = material.getParameter("specSize");
            var refC = material.getParameter("reflColor");
            var refT = material.getParameter("reflTex");
            var refIntensity = material.getParameter("reflIntensity");
            var refMix = material.getParameter("reflMix");
            var emitC = material.getParameter("emissionColor");
            var bumpT = material.getParameter("bumpTex");
            var bumpIntensity = material.getParameter("bumpIntensity");
            var transC = material.getParameter("transColor");
            var transT = material.getParameter("transTex");
            var transIntensity = material.getParameter("transIntensity");
            var transMix = material.getParameter("transMix");
            var fresnel = material.getParameter("transFresnel");

            if (!colorT) colorMix = 1;
            if (!specT) specMix = 1;
            if (!refT) refMix = 1;
            if (!transT) transMix = 1;
            
            var ior = (mat_ior_e != undefined || mat_ior_e > 0)? mat_ior_e : material.getParameter("transEta");
            // diffuse calculation
            colorC = colorC.multiply(defIntensity);
            // specular calculation
            specSize = (128 - specSize) / 128 * 1000; // 0-1000
            //
            if (mode == 'block' || mode == 'block_extra') {
                mat_name = matName('mat',mat_name);
                
                if (mode == 'block') {
                    if (colorT != 'none') colorN = writeImageBlock(SCN_FILE, colorT);
                    if (transT != 'none') transN = writeImageBlock(SCN_FILE, transT);
                    if (refT != 'none') refN = writeImageBlock(SCN_FILE, refT);
                    if (specT != 'none') specN = writeImageBlock(SCN_FILE, specT);
                    if (bumpT != 'none') bumpN = writeImageBlock(SCN_FILE, bumpT);
                }
            }
            
            if (mode == 'block' || mode == 'block_extra') SCN_FILE.writeln("material "+mat_name+" = {");
            else SCN_FILE.writeln("\tmaterial {"); // inline
            
            colorC = colorC.multiply(colorMix);
            if (colorC.norm() > 0) SCN_FILE.writeln(indent+"\tdiffuse "+vector3Line(colorC));
            if (colorT != 'none') SCN_FILE.writeln(indent+"\tmap image {"+colorN+matrix_line+"}");
            
            if (transIntensity > 0) { // transmittance material
                SCN_FILE.writeln(indent+"\ttransmit "+vector3Line(transC.multiply(transIntensity*transMix)));
                if (transT != 'none') SCN_FILE.writeln(indent+"\tmap image {"+transN+matrix_line+"}");
            }
            if (fresnel) { // if fresnel is On, use reflect 0.
                SCN_FILE.writeln("\treflect 0");
                if (refT != 'none') SCN_FILE.writeln(indent+"\tmap image {"+refN+matrix_line+"}");
            } else if (refIntensity > 0) {
                SCN_FILE.writeln(indent+"\treflect "+vector3Line(refC.multiply(refIntensity*refMix)));
                if (refT != 'none') SCN_FILE.writeln(indent+"\tmap image {"+refN+matrix_line+"}");
            }
            if (specSize > 0) {
                SCN_FILE.writeln(indent+"\tspecular "+vector3Line(specC.multiply(specMix))+" "+parseInt(specSize));
                if (specT != 'none') SCN_FILE.writeln(indent+"\tmap image {"+specN+matrix_line+"}");
            }
            // emission
            if (emitC.r > 0 || emitC.g > 0 || emitC.b > 0) SCN_FILE.writeln(indent+"\temit "+vector3Line(emitC));

            if (bumpT != 'none') {
                SCN_FILE.writeln(indent+"\tbump_map image {"+bumpN+" hscale "+bumpIntensity.toPrecision(6)+matrix_line+" }");
            } else if (bumpIntensity > 1) {
                var exponent = bumpIntensity * 20; // 2.5 -> 50, 5.0 -> 100
                SCN_FILE.writeln(indent+"\trough 0.000001 50/"+exponent); //?
            }
            
            if (ior > 1) SCN_FILE.writeln(indent+"\tior "+ior.toPrecision(6));
            
            SCN_FILE.writeln(indent+"}\n");
            break;
        case "SolidColor":
            var colorC = material.getParameter("colorColor");
            var colorT = material.getParameter("colorTex");
            var colorIntensity = material.getParameter("colorIntensity");
            colorC = colorC.multiply(colorIntensity);
            if (mode == 'block' || mode == 'block_extra') {
                mat_name = matName('emit',mat_name);
                if (colorT != 'none') colorN = writeImageBlock(SCN_FILE, colorT);
                SCN_FILE.writeln("material "+mat_name+" = {");
            } else {
                SCN_FILE.writeln("\tmaterial {");
            }
            SCN_FILE.writeln(indent+"\temit "+vector3Line(colorC));
            if (colorT != 'none') SCN_FILE.writeln(indent+"\tmap image {"+colorN+matrix_line+"}");
            
            SCN_FILE.writeln(indent+"}\n");
            break;
        case "Wood": // Wood for procedural diffuse layer.
        case "Marble":
        case "Directional":
            if (mat_type == "Wood") {
                var color1 = material.getParameter("fairColor");
                var color2 = material.getParameter("darkColor");
                var refIntensity = 0;
            } else if (mat_type == "Marble") {
                var color1 = material.getParameter("fairColor");
                var color2 = material.getParameter("darkColor");
                var refIntensity = material.getParameter("reflIntensity");
            } else {
                var color1 = material.getParameter("XColor");
                var color2 = material.getParameter("YColor");
                var color3 = material.getParameter("ZColor");
                var refIntensity = 0;
            }
            var m_name = material.getParameter("name");
            var m_parameters = m_name.split("_"); // 1 -> type, 2 -> mappingType, 3 -> radius, 4 -> diffuse, 5 -> ior
            var m_ior = (m_parameters.length < 6)? 1.0 : parseFloat(m_parameters[5]);
            var m_diffuse = (m_parameters.length < 5)? 1 : parseFloat(m_parameters[4]);
            var m_radius = (m_parameters.length < 4)? 0.5 : parseFloat(m_parameters[3]);
            var m_maptype = (m_parameters.length < 3)? 1 : parseInt(m_parameters[2]);
            var m_type = (m_parameters.length < 2)? "noise" : m_parameters[1];
            var mat_name = m_parameters[0];
            var specC = material.getParameter("specColor");
            var specSize = material.getParameter("specSize");
            specSize = (128 - specSize) / 128 * 1000;
            if (Procedurals[m_type] == undefined) m_type = 'noise';
            if (mode == 'block' || mode == 'block_extra') {
                mat_name = matName('pro',mat_name);
                SCN_FILE.writeln("material "+mat_name+" = {");
            } else {
                SCN_FILE.writeln("\tmaterial {"); // inline
            }
            SCN_FILE.writeln(indent+"\tdiffuse "+m_diffuse);
            SCN_FILE.writeln(indent+"\tmap "+m_type+" {");
            if (Procedurals[m_type]) SCN_FILE.writeln(indent+"\t\t"+m_radius);
            if (mat_type != "Directional") {
                SCN_FILE.writeln(indent+"\t\tcolour_band 0.0 0.5 "+vector3Line(color1)+" "+vector3Line(color2));
                SCN_FILE.writeln(indent+"\t\tcolour_band 0.5 1.0 "+vector3Line(color2)+" "+vector3Line(color2));
            } else {
                SCN_FILE.writeln(indent+"\t\tcolour_band 0.0 0.33 "+vector3Line(color1)+" "+vector3Line(color2));
                SCN_FILE.writeln(indent+"\t\tcolour_band 0.33 0.66 "+vector3Line(color2)+" "+vector3Line(color3));
                SCN_FILE.writeln(indent+"\t\tcolour_band 0.66 1.0 "+vector3Line(color3)+" "+vector3Line(color3));
            }
            if (mode == 'inline' || mode == 'block_extra') {
                if (m_maptype == 1) writeMatrix(SCN_FILE, info[3], info[4], info[5]);
                else SCN_FILE.writeln(indent+"\t\t"+matrix_line);
            }
            SCN_FILE.writeln(indent+"\t\tmaptype "+m_maptype);          
            SCN_FILE.writeln(indent+"\t}");
            if (specSize > 0) {
                SCN_FILE.writeln(indent+"\tspecular "+vector3Line(specC)+" "+parseInt(specSize));
            }
            if (refIntensity > 0) {
                SCN_FILE.writeln(indent+"\treflect "+refIntensity.toPrecision(4));
            }
            if (m_ior > 1) {
                SCN_FILE.writeln(indent+"\tior "+m_ior.toPrecision(3));
            }
            SCN_FILE.writeln(indent+"}\n");
            break;
        default: // dummy material.
            mat_name = matName('ud',mat_name);
            if (mode == 'block' || mode == 'block_extra') SCN_FILE.writeln("material "+mat_name+" = {");
            else SCN_FILE.writeln("\tmaterial {");
            SCN_FILE.writeln(indent+"\tdiffuse 0.8");
            SCN_FILE.writeln(indent+"}\n");
            break;
    }
    
    return mat_name;
}
// debug functions.
function printVec3D(str, vec) {
    print(str + ':x:'+vec.x.toPrecision(3)+', y:'+vec.y.toPrecision(3)+', z:'+vec.z.toPrecision(3));
}
function printVec4D(vec) {
    print('r:g:b:a-'+vec.r+':'+vec.g+':'+vec.b+':'+vec.a);
}
function printMatrix(matrix) {
    print(matrix.m00.toPrecision(3)+':'+matrix.m01.toPrecision(3)+':'+matrix.m02.toPrecision(3)+':'+matrix.m03.toPrecision(3));
    print(matrix.m10.toPrecision(3)+':'+matrix.m11.toPrecision(3)+':'+matrix.m12.toPrecision(3)+':'+matrix.m13.toPrecision(3));
    print(matrix.m20.toPrecision(3)+':'+matrix.m21.toPrecision(3)+':'+matrix.m22.toPrecision(3)+':'+matrix.m23.toPrecision(3));
    print(matrix.m30.toPrecision(3)+':'+matrix.m31.toPrecision(3)+':'+matrix.m32.toPrecision(3)+':'+matrix.m33.toPrecision(3));    
}
function printParameterInfo(obj) {
    var i;
    var objInfo = obj.parameterInfo();
    print("--- parameterInfo ---");
    for (i = 0;i < objInfo.length;i++) {
        print(objInfo[i][0]+"  ["+objInfo[i][1]+"]");
        print(obj.getParameter(objInfo[i][0]));
    }
}
